# --------------------------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for license information.
# --------------------------------------------------------------------------------------------

import unittest

from azure.cli.testsdk import ScenarioTest, ResourceGroupPreparer


class CognitiveServicesPropertiesTests(ScenarioTest):
    @ResourceGroupPreparer()
    def test_cognitiveservices_account_capabilities(self, resource_group):
        sname = self.create_random_name(prefix='cs_cli_test_', length=16)

        self.kwargs.update({
            'sname': sname,
            'kind': 'FormRecognizer',
            'sku': 'S0',
            'location': 'SOUTHCENTRALUS'
        })

        # test to create cognitive services account
        self.cmd('az cognitiveservices account create -n {sname} -g {rg} --kind {kind} --sku {sku} -l {location} --yes',
                 checks=[self.check('name', '{sname}'),
                         self.check('location', '{location}'),
                         self.check('sku.name', '{sku}')])
        account = self.cmd('az cognitiveservices account show -n {sname} -g {rg}').get_output_in_json()
        self.assertTrue(account['properties']['capabilities'] is not None)
        self.assertTrue(len(account['properties']['capabilities']) > 0)
        self.assertTrue(account['properties']['capabilities'][0]['name'] is not None)
        self.assertTrue(account['properties']['capabilities'][0]['name'] != '')

        # delete the cognitive services account
        ret = self.cmd('az cognitiveservices account delete -n {sname} -g {rg}')
        self.assertEqual(ret.exit_code, 0)

    @ResourceGroupPreparer()
    def test_cognitiveservices_account_public_network_access(self, resource_group):
        sname = self.create_random_name(prefix='cs_cli_test_', length=16)

        self.kwargs.update({
            'sname': sname,
            'kind': 'Face',
            'sku': 'S0',
            'location': 'SOUTHCENTRALUS'
        })

        # test to create cognitive services account
        self.cmd('az cognitiveservices account create -n {sname} -g {rg} --kind {kind} --sku {sku} -l {location} --yes',
                 checks=[self.check('name', '{sname}'),
                         self.check('location', '{location}'),
                         self.check('sku.name', '{sku}')])
        account = self.cmd('az cognitiveservices account show -n {sname} -g {rg}').get_output_in_json()
        print(account)
        self.assertEqual(account['properties']['publicNetworkAccess'], 'Enabled')


        # delete the cognitive services account
        ret = self.cmd('az cognitiveservices account delete -n {sname} -g {rg}')
        self.assertEqual(ret.exit_code, 0)

    @ResourceGroupPreparer()
    def test_cognitiveservices_aiservices(self, resource_group):
        sname = self.create_random_name(prefix='csclitest_', length=16)
        sname2 = self.create_random_name(prefix='csclitest2_', length=16)
        self.kwargs.update({
            'sname': sname,
            'sname2': sname2,
            'kind': 'AIServices',
            'sku': 'S0',
            'location': 'EASTUS'
        })

        # test to create cognitive services account with allow_project_management set to True by default for AIServices kind
        self.cmd('az cognitiveservices account create -n {sname} -g {rg} --kind {kind} --sku {sku} -l {location} --yes',
                 checks=[self.check('name', '{sname}'),
                         self.check('location', '{location}'),
                         self.check('sku.name', '{sku}'),
                         self.check('properties.allowProjectManagement', True)])
        account = self.cmd('az cognitiveservices account show -n {sname} -g {rg}').get_output_in_json()
        self.assertTrue(account['identity']['type'] == 'SystemAssigned')

        # delete the cognitive services account
        ret = self.cmd('az cognitiveservices account delete -n {sname} -g {rg}')
        self.assertEqual(ret.exit_code, 0)

        self.cmd('az cognitiveservices account create -n {sname2} -g {rg} --kind {kind} --sku {sku} -l {location} --allow-project-management true --yes',
                 checks=[self.check('name', '{sname2}'),
                         self.check('location', '{location}'),
                         self.check('sku.name', '{sku}'),
                         self.check('properties.allowProjectManagement', True)])
        account = self.cmd('az cognitiveservices account show -n {sname2} -g {rg}').get_output_in_json()
        self.assertTrue(account['identity']['type'] == 'SystemAssigned')

        # delete the cognitive services account
        ret = self.cmd('az cognitiveservices account delete -n {sname2} -g {rg}')
        self.assertEqual(ret.exit_code, 0)


if __name__ == '__main__':
    unittest.main()
